/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_06;
//ESCA*JAVA0267
//ESCA*JAVA0076

import java.io.File;
import java.io.PrintWriter;
import java.util.Random; import java.util.Vector;

import djbuch.kapitel_13.DJCacheDocument;

import lotus.domino.*;

/**
 * Kleine Kollektion der verschiedensten Hilfsfunktionen.
 * Zum Allgemeingltiger Art, wie z.B. dumpDoc (Document) - Ausgabe eines Domino Document
 * auf die Console oder einen Printwriter - oder printNavigator - Ausgabe eines
 * ViewNavigator -, zum Teil spezifischer Art, als Hilfsmethode fr diese Code-Sammlung,
 * wie z.b. startup und cleanup
 *
 * @author Thomas Ekert
 */
public class DemoTools {

	private static final int NUM_OF_DOC_ITEMS = 10;		// Anzahl von (dummy) Items in den Dokumenten
	private static final int SIZE_OF_DOC_ITEMS = 100;	// Datengre in diesen Items in Byte.
	
	public static final boolean RANDOM_INDEX = false; 	// Sollen Dokumente der
															// Art XXX0...XXX999 oder
														  	// mit zuflligem Index
														  	// XXXKLM...XXXOPQ erzeugt
														  	// werden?
	private static final boolean FULLTEXT_INDEX = true;
	
	private static final Random generator = new Random(System.currentTimeMillis());
	public static final String ANIMAL_A = "Tier ";
	private static final String ANIMAL_B = "Ambe ";
	protected static final String [] CATEGORIES = {
			"Wombats", "Hunde", "Katzen", "Muse", "Tiere\\Gro", "Tiere\\Klein", "Esel", "Knigsberger Tigerforellen"
	};
	
	/**
	 * Prft, ob ein String leer ist.
	 */
	public static final boolean isEmpty(String s)
	{
		if (s == null) { return true; }
		return (s.equals(""));
	}
	
	/**
	 * @see getServerName (Session, boolean)
	 * @param mSession
	 * @return - getServerName (mSession, false)
	 */
	public static final String getServerName(Session mSession) {
		return getServerName (mSession, false);
	}
	
	/**
	 * liefert den Servernamen des tatschlichen Servers auf dem die Session luft.
	 * Im Gegensatz zu session.getServerName, liefert diese Methode auch den korrekten
	 * Servernamen (und nicht ""), auf einer lokalen serverbasierten Session.
	 * @param mSession
	 * @param abbreviated - falls true, wird der Name vereinfacht, also ohne das CN= usw. zurckgegeben.
	 * @return - den kanonischen oder vereinfachten (abbreviated) Servernamen des Servers, auf den die Session sich bezieht oder "Exception_noServerFound" im Fehlerfall.
	 */
	public static final String getServerName(Session mSession, boolean abbreviated) {
		try {
			String servername = null;
			Name n = mSession.getUserNameObject();
			if (n.isHierarchical()) {
				if (abbreviated) {
					servername = n.getAbbreviated();
				} else {
					servername = n.getCanonical();
				}
			} else {
				System.err.println ("Konnte den Servernamen nicht kanonisch aufloesen.");
			}
			return servername;
		}
		catch (NotesException e) {
			e.printStackTrace();
			return "Exception_noServerFound";
		}
	}

	/** Erzeugt numberOfDocs * 1,1 neue Dokumente unterschiedlichen Inhalts
	 * fr Testzwecke
	 * @param db
	 * @param numberOfDocs
	 * @throws NotesException
	 */
	public static void startup (Database db, int numberOfDocs) throws NotesException {
		cleanup(db);
		Document doc = null;
		try {
			System.out.println ("Erzeuge " + (numberOfDocs + (numberOfDocs/100)) + " neue Dokumente.");
			for (int i = 0; i<numberOfDocs; i++) {
				String index = RANDOM_INDEX?getRandom(7):""+i;
				doc = createDoc(db,CATEGORIES[i % CATEGORIES.length],ANIMAL_A + index,null);
				GC.recycle(doc);
			}
			for (int i = 0; i<numberOfDocs/100; i++) {
				doc = createDoc(db,CATEGORIES[i % CATEGORIES.length],ANIMAL_B + i,null);
				GC.recycle(doc);
			}
			if (FULLTEXT_INDEX) {
				System.out.println ("FT Index aktualisieren.");
				db.updateFTIndex(true);
			}
		} finally {
			GC.recycle(doc);
		}
	}
	
	/**
	 * @see createDoc (Database, String, String, String, Document)
	 * @param db
	 * @param cat
	 * @param title
	 * @param parent
	 * @return
	 * @throws NotesException
	 */
	public static Document createDoc (Database db, String cat, String title, Document parent) throws NotesException {
		return (createDoc(db,"FO_dokument_k6",cat,title,parent));
	}
	
	/**
	 * Erzeugt ein neues (Test-) Dokument. Das Dokument wird durch das Item
	 * F_marker_k6=1 markiert, damit es in cleanup wieder gelcht werden kann.
	 * @see cleanup (Database)
	 * @param db - Datenbank, in der das Document erstellt werden soll,
	 * @param form - Maskenname fr dieses Document (item Form)
	 * @param cat - Inhalt des Items "F_Category"
	 * @param title - Inhalt des Items "F_titel"
	 * @param parent - Optionales Mutterdokument (dieses neue Doc wird
	 * 		Antwortdokument). Darf auch null sein.
	 * @return - neue Document
	 * @throws NotesException
	 */
	public static Document createDoc (Database db, String form, String cat, String title, Document parent) throws NotesException {
		Document doc = db.createDocument();
		doc.replaceItemValue ("Form", form);
		doc.replaceItemValue ("F_Category", cat);
		doc.replaceItemValue ("F_titel", title);
		//Anhand dieses Markers kann erkannt werden, dass createDoc ein Dokument erstellt hat.
		//Wird frs Aufrumen bentigt - siehe cleanup.
		doc.replaceItemValue ("F_marker_k6",new Integer (1));
		for (int i = 0; i < NUM_OF_DOC_ITEMS; i++) {
			doc.replaceItemValue ("Item_" + i, getRandomString(SIZE_OF_DOC_ITEMS));
		}
		if (parent != null) {
			doc.makeResponse(parent);
		}
		doc.save (true,false);
		return doc;
	}
	
	/**
	 * gibt ein Document als Name-Value Paare auf die Console aus.
	 * @param doc
	 * @throws NotesException
	 */
	public static final void dumpDoc (Document doc) throws NotesException {
		Vector v = doc.getItems();
		for (int i=0, s=v.size(); i<s; i++) {
			Item item = (Item)v.elementAt(i);
			System.out.print (" " + item.getName()+"=");
			Vector val=item.getValues();
			System.out.println (val!=null?val.toString():"null");
		}
	}
	
	/**
	 * Gibt ein Document als Name-Value Paare auf einen PrintWriter aus.
	 * @param doc
	 * @param out
	 * @throws NotesException
	 */
	public static final void dumpDoc (Document doc, PrintWriter out) throws NotesException {
		Vector v = doc.getItems();
		for (int i=0, s=v.size(); i<s; i++) {
			Item item = (Item)v.elementAt(i);
			out.print (" " + item.getName()+"=");
			Vector val=item.getValues();
			out.println (val!=null?val.toString():"null");
		}
	}

	/** berprft, ob unter dem angegebenen pfadnamen "fileName" eine Datei existiert.
	 * 
	 * @param fileName
	 * @return
	 */
	public static boolean exists (String fileName) {
		File f = new File(fileName);
		return (f.exists());
	}
	
	/** prft ob ein gegebener dirName ein Directory reprsentiert und erzeugt gegebenenfalls ein solches dir */
	public static boolean makeDir (String dirName) {
		File d = new File (dirName);
		if (d.exists() && !d.isDirectory()) {
			return false;
		}
		if (d.exists() && d.isDirectory()) {
			return true;
		}
		return d.mkdirs ();
	}
	/**
	 * Lscht alle Dokumente der Datenbank <b>db</b>, fr die F_marker_k6=1.
	 * @param db
	 * @throws NotesException
	 */
	private static final void cleanup (Database db) throws NotesException {
		DocumentCollection col = null;
		try {
			col = db.search ("F_marker_k6=1");
			System.out.println ("Aufrumen: Lsche " + col.getCount() + " Dokumente.");
			col.removeAll(true);
		} finally {
			if (col!=null) {
				col.recycle();
			}
		}
	}
	/**
	 * Gibt die 10er Potenz von <b>number</b> zurck.
	 * @param number
	 * @return
	 */
	public static final int whichPowerOfTen (int number) {
		for (int k = 0; k < 10; k++) {
			if (Math.pow (10,k) > number) {
				return k-1;
			}
		}
		return 0;
	}
	
	public static final String repeatChar (char c, int count) {
		StringBuffer sb = new StringBuffer();
		for (int k=0;k<count;k++) {
			sb.append(c);
		}
		return sb.toString();
	}
	
	public static final String getRandom (int length) {
		double tenPow = Math.pow (10,length);
		double tenPowMinus = Math.pow (10,length-1);
		double d = tenPow*(generator.nextDouble());
		if (d < (tenPowMinus)) {
			d += tenPowMinus;
		}
		return ""+Math.round(d);
	}

	/**
	 * generiert einen <b>i</b>-stelligen ZufallsString aus a..z, A..Z, 0..9
	 * @param i
	 * @return
	 */
	public static String getRandomString(int i)
	{
	    char[] ac = {
	        'a', 'b', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 
	        'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 
	        't', 'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 
	        'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 
	        'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 
	        'X', 'Y', 'Z', '0', '1', '2', '3', '4', '5', '6',
			'6', '7', '8', '9'
	    };
	    StringBuffer stringbuffer = new StringBuffer();
	    for(int j = 0; j < i; j++) {
	        stringbuffer.append(ac[Math.abs(generator.nextInt()) % ac.length]);
	    }
	
	    return stringbuffer.toString();
	}
	
	public static int getRandomNumber () {
		return generator.nextInt();
	}
	
	/** 
	 * Zufallszahl zwischen <b>min</b> einschlielich und <b>max</b> einschlielich.
	 * @param min
	 * @param max
	 * @return
	 */
	public static int getRandomNumber (int min, int max) {
		return min + generator.nextInt (max+1);
	}

	/**
	 * String Replacement Funktion fr JVM <= 1.3.x
	 * @param source
	 * @param from
	 * @param to
	 * @return
	 */
	public static String replaceSubstring(String source, String from, String to) {
		if (source == null) {
			return null;
		}
		StringBuffer result = new StringBuffer();
		int i = 0;
		for (int j = 0; (j = source.indexOf(from, i)) >= 0;) {
			result.append(source.substring(i, j));
			result.append(to);
			i = j + from.length();
		}
	
		result.append(source.substring(i));
		return result.toString();
	}

	/**
	 * @param djDoc
	 */
	public static void dumpDoc(DJCacheDocument djDoc) {
		System.out.println (djDoc.toString());
	}

	/**
	 * Einfache Ausgabe Routine fr ViewNavigator
	 * @param nav
	 */
	public static final void printNavigator (ViewNavigator nav) {
		ViewEntry entry = null, nextEntry = null;
		try {
			entry = nav.getFirst();
			while (entry != null) {
				if (entry.isDocument()) {
					System.out.println ("\tDokument: " + entry.getDocument().getItemValueString("F_titel"));
				}
				if (entry.isCategory()) {
					System.out.println ("Kategorie: " + entry.getColumnValues().elementAt(0));
				}
				if (entry.isTotal()) {
					System.out.println ("Total: " + entry.getColumnValues().elementAt(3));
				}
				if (entry.isConflict()) {
					System.out.println ("\t\tSpeicherkonflikt: " + entry.getDocument().getItemValueString("F_titel"));
				}
				nextEntry = nav.getNext();
				GC.recycle(entry);
				entry=nextEntry;
			}
		} catch (NotesException e) {
			e.printStackTrace();
		} finally {
			GC.recycle (entry);
			GC.recycle (nextEntry);
		}
	}
	
	private DemoTools () {
		//hide me
	}
}
